<?php
/**
 * Multi Account
 * https://webenginecms.org/
 * 
 * @version 1.0.1
 * @author Lautaro Angelico <http://lautaroangelico.com/>
 * @copyright (c) 2013-2019 Lautaro Angelico, All Rights Reserved
 * @build aa08b26fa92cd07a3c15ac4a872be8e7
 */

namespace Plugin\MultiAccount;

class MultiAccount extends \Account {
	
	private $_modulesPath = 'modules';
	private $_defaultAccountSerial = '1111111111111';
	
	// CONSTRUCTOR
	
	function __construct() {
		parent::__construct();
	}
	
	// PUBLIC FUNCTIONS
	
	public function registerAccount($username, $password, $cpassword, $email) {
		
		if(!check_value($username)) throw new \Exception(lang('error_4',true));
		if(!check_value($password)) throw new \Exception(lang('error_4',true));
		if(!check_value($cpassword)) throw new \Exception(lang('error_4',true));
		if(!check_value($email)) throw new \Exception(lang('error_4',true));

		// Filters
		if(!\Validator::UsernameLength($username)) throw new \Exception(lang('error_5',true));
		if(!\Validator::AlphaNumeric($username)) throw new \Exception(lang('error_6',true));
		if(!\Validator::PasswordLength($password)) throw new \Exception(lang('error_7',true));
		if($password != $cpassword) throw new \Exception(lang('error_8',true));
		if(!\Validator::Email($email)) throw new \Exception(lang('error_9',true));
		
		# load registration configs
		$regCfg = loadConfigurations('register');
		
		# check if username / email exists
		if($this->userExists($username)) throw new \Exception(lang('error_10',true));
		
		# WebEngine Email Verification System (EVS)
		if($regCfg['verify_email']) {
			
			# check if username / email exists
			if($this->checkUsernameEVS($username)) throw new \Exception(lang('error_10',true));
			
			# generate verification key
			$verificationKey = $this->createRegistrationVerification($username,$password,$email);
			if(!check_value($verificationKey)) throw new \Exception(lang('error_23',true));
			
			# send verification email
			$this->sendRegistrationVerificationEmail($username,$email,$verificationKey);
			message('success', lang('success_18',true));
			return;
		}
		
		# insert data
		$data = array(
			'username' => $username,
			'password' => $password,
			'name' => $username,
			'serial' => $this->_defaultAccountSerial,
			'email' => $email
		);
		
		# query
		if($this->_md5Enabled) {
			$query = "INSERT INTO "._TBL_MI_." ("._CLMN_USERNM_.", "._CLMN_PASSWD_.", "._CLMN_MEMBNAME_.", "._CLMN_SNONUMBER_.", "._CLMN_EMAIL_.", "._CLMN_BLOCCODE_.", "._CLMN_CTLCODE_.") VALUES (:username, [dbo].[fn_md5](:password, :username), :name, :serial, :email, 0, 0)";
		} else {
			$query = "INSERT INTO "._TBL_MI_." ("._CLMN_USERNM_.", "._CLMN_PASSWD_.", "._CLMN_MEMBNAME_.", "._CLMN_SNONUMBER_.", "._CLMN_EMAIL_.", "._CLMN_BLOCCODE_.", "._CLMN_CTLCODE_.") VALUES (:username, :password, :name, :serial, :email, 0, 0)";
		}
		
		# register account
		$result = $this->memuonline->query($query, $data);
		if(!$result) throw new \Exception(lang('error_22',true));
		
		# send welcome email
		if($regCfg['send_welcome_email']) {
			$this->sendWelcomeEmail($username, $email);
		}
		
		# success message
		message('success', lang('success_1',true));
		
		# redirect to login (5 seconds)
		redirect(2,'login/',5);
	}
	
	public function passwordRecoveryProcess($username, $user_email, $ip_address) {
		if(!check_value($username)) throw new \Exception(lang('error_4',true));
		if(!\Validator::UsernameLength($username)) throw new \Exception(lang('error_5',true));
		if(!\Validator::AlphaNumeric($username)) throw new \Exception(lang('error_6',true));
		if(!check_value($user_email)) throw new \Exception(lang('error_30',true));
		if(!check_value($ip_address)) throw new \Exception(lang('error_30',true));
		if(!\Validator::Email($user_email)) throw new \Exception(lang('error_30',true));
		if(!\Validator::Ip($ip_address)) throw new \Exception(lang('error_30',true));
		
		if(!$this->emailExists($user_email)) throw new \Exception(lang('error_30',true));
		
		$user_id = $this->retrieveUserID($username);
		if(!check_value($user_id)) throw new \Exception(lang('error_23',true));
		
		$accountData = $this->accountInformation($user_id);
		if(!is_array($accountData)) throw new \Exception(lang('error_23',true));
		
		# Account Recovery Code
		$arc = $this->generateAccountRecoveryCode($accountData[_CLMN_MEMBID_], $accountData[_CLMN_USERNM_]);

		# Account Recovery URL
		$aru = $this->generateAccountRecoveryLink($accountData[_CLMN_MEMBID_], $accountData[_CLMN_EMAIL_], $arc);
		
		# send email
		try {
			$email = new \Email();
			$email->setTemplate('PASSWORD_RECOVERY_REQUEST');
			$email->addVariable('{USERNAME}', $accountData[_CLMN_USERNM_]);
			$email->addVariable('{DATE}', date("Y-m-d @ h:i a"));
			$email->addVariable('{IP_ADDRESS}', $ip_address);
			$email->addVariable('{LINK}', $aru);
			$email->addAddress($accountData[_CLMN_EMAIL_]);
			$email->send();
			
			message('success', lang('success_6',true));
		} catch (\Exception $ex) {
			if($this->_debug) {
				throw new \Exception($ex->getMessage());
			} else {
				throw new \Exception(lang('error_23',true));
			}
		}
	}
	
	public function passwordRecoveryVerificationProcess($ui, $ue, $key) {
		if(!check_value($ui)) throw new \Exception(lang('error_31',true));
		if(!check_value($ue)) throw new \Exception(lang('error_31',true));
		if(!check_value($key)) throw new \Exception(lang('error_31',true));
		
		$user_id = Decode($ui); // decoded user id
		if(!\Validator::UnsignedNumber($user_id)) throw new \Exception(lang('error_31',true));
		
		$user_email = Decode($ue); // decoded email address
		if(!$this->emailExists($user_email)) throw new \Exception(lang('error_31',true));
		
		$accountData = $this->accountInformation($user_id);
		if(!is_array($accountData)) throw new \Exception(lang('error_31',true));
		
		$username = $accountData[_CLMN_USERNM_];
		$gen_key = $this->generateAccountRecoveryCode($user_id, $username);
		
		# compare keys
		if($key != $gen_key) throw new \Exception(lang('error_31',true));
		
		# update user password
		$new_password = rand(11111111,99999999);
		$update_pass = $this->changePassword($user_id, $username, $new_password);
		if(!$update_pass) throw new \Exception(lang('error_23',true));

		try {
			$email = new \Email();
			$email->setTemplate('PASSWORD_RECOVERY_COMPLETED');
			$email->addVariable('{USERNAME}', $username);
			$email->addVariable('{NEW_PASSWORD}', $new_password);
			$email->addAddress($accountData[_CLMN_EMAIL_]);
			$email->send();
			
			message('success', lang('success_7',true));
		} catch (\Exception $ex) {
			if($this->_debug) {
				throw new \Exception($ex->getMessage());
			} else {
				throw new \Exception(lang('error_23',true));
			}
		}
	}
	
	public function loadModule($module) {
		if(!\Validator::Alpha($module)) throw new \Exception('Could not load plugin module (multiaccount)');
		if(!$this->_moduleExists($module)) throw new \Exception('Could not load plugin module (multiaccount)');
		if(!@include_once(__PATH_MULTIACCOUNT_ROOT__ . $this->_modulesPath . '/' . $module . '.php')) throw new \Exception('Could not load plugin module (multiaccount)');
	}
	
	// PRIVATE FUNCTIONS
	
	private function checkUsernameEVS($username) {
		if(!check_value($username)) return;
		$result = $this->memuonline->query_fetch_single("SELECT * FROM ".WEBENGINE_REGISTER_ACCOUNT." WHERE registration_account = ?", array($username));
		
		$configs = loadConfigurations('register');
		if(!is_array($configs)) return;
		
		$timelimit = $result['registration_date']+$configs['verification_timelimit']*60*60;
		if($timelimit > time()) return true;
		
		$this->deleteRegistrationVerification($username);
		return false;
	}
	
	private function createRegistrationVerification($username,$password,$email) {
		if(!check_value($username)) return;
		if(!check_value($password)) return;
		if(!check_value($email)) return;
		
		$key = uniqid();
		$data = array(
			$username,
			$password,
			$email,
			time(),
			$_SERVER['REMOTE_ADDR'],
			$key
		);
		
		$query = "INSERT INTO ".WEBENGINE_REGISTER_ACCOUNT." (registration_account,registration_password,registration_email,registration_date,registration_ip,registration_key) VALUES (?,?,?,?,?,?)";
		
		$result = $this->memuonline->query($query, $data);
		if(!$result) return;
		return $key;
	}
	
	private function sendRegistrationVerificationEmail($username, $account_email, $key) {
		$verificationLink = __BASE_URL__.'verifyemail/?op=2&user='.$username.'&key='.$key;
		try {
			$email = new \Email();
			$email->setTemplate('WELCOME_EMAIL_VERIFICATION');
			$email->addVariable('{USERNAME}', $username);
			$email->addVariable('{LINK}', $verificationLink);
			$email->addAddress($account_email);
			$email->send();
		} catch (\Exception $ex) {
			if($this->_debug) {
				throw new \Exception($ex->getMessage());
			}
		}
	}
	
	private function sendWelcomeEmail($username,$address) {
		try {
			$email = new \Email();
			$email->setTemplate('WELCOME_EMAIL');
			$email->addVariable('{USERNAME}', $username);
			$email->addAddress($address);
			$email->send();
		} catch (\Exception $ex) {
			if($this->_debug) {
				throw new \Exception($ex->getMessage());
			}
		}
	}
	
	private function generateAccountRecoveryLink($userid,$email,$recovery_code) {
		if(!check_value($userid)) return;
		if(!check_value($recovery_code)) return;
		
		$build_url = __BASE_URL__;
		$build_url .= 'forgotpassword/';
		$build_url .= '?ui=';
		$build_url .= $userid;
		$build_url .= '&ue=';
		$build_url .= $email;
		$build_url .= '&key=';
		$build_url .= $recovery_code;
		return $build_url;
	}
	
	private function _moduleExists($module) {
		if(!check_value($module)) return;
		if(!file_exists(__PATH_MULTIACCOUNT_ROOT__ . $this->_modulesPath . '/' . $module . '.php')) return;
		return true;
	}
	
	private function deleteRegistrationVerification($username) {
		if(!check_value($username)) return;
		$delete = $this->memuonline->query("DELETE FROM ".WEBENGINE_REGISTER_ACCOUNT." WHERE registration_account = ?", array($username));
		if($delete) return true;
		return;
	}
}